import { PortableText } from "next-sanity";
import imageUrlBuilder from "@sanity/image-url";
import { client } from "@/sanity/lib/client";
import Link from "next/link";
import { Navbar } from "@/components/Navbar";
import Footer from "@/components/Footer";
import styles from "./styles.module.scss"

const POST_QUERY = `
  *[_type == "post" && slug.current == $slug][0]{
    title,
    slug,
    publishedAt,
    author-> {name},
    categories[]->{_id, title},
    mainImage {asset->{_ref, url}, alt},
    media[]{_type, asset->{_ref, url}, alt},
    body
  }
`;


const { projectId, dataset } = client.config();
const urlFor = (source) =>
    projectId && dataset
        ? imageUrlBuilder({ projectId, dataset }).image(source)
        : null;

const options = { next: { revalidate: 30 } };

export default async function PostPage({ params }) {
    const post = await client.fetch(POST_QUERY, params, options);
    console.log(post);

    return (
        <>
            <Navbar />
            <div className="flex flex-col items-center justify-center bg-slate-950 h-80">
                <div className='h-32 w-full'>

                </div>
                <h1 className="lg:text-8xl md:text-5xl text-4xl font-bold text-white " >{post.title}</h1>

            </div>
            <main className="container mx-auto min-h-screen max-w-3xl p-8 flex flex-col gap-4">


                <Link href="/blog" className="hover:underline">
                    ← Back to posts
                </Link>

                {/* Media Section: Handle images and videos */}
                {post.media && post.media.length > 0 && (
                    <div className="media-section mb-6">
                        {post.media.map((item, index) => {
                            if (item._type === "image") {
                                const imageUrl = urlFor(item.asset).width(550).height(310).url();
                                return (
                                    <img
                                        key={index}
                                        src={imageUrl}
                                        alt={item.alt || post.title}
                                        className="aspect-video rounded-xl"
                                        width="550"
                                        height="310"
                                    />
                                );
                            } else if (item._type === "file") {
                                return (
                                    <video key={index} controls className="w-full rounded-xl">
                                        <source src={item.asset.url} type="video/mp4" />
                                        Your browser does not support the video tag.
                                    </video>
                                );
                            }
                        })}
                    </div>
                )}

                <h1 className="text-4xl font-bold mb-4">{post.title}</h1>

                {/* Meta Information */}
                <div className="text-gray-500 text-sm mb-8">
                    <p>Published: {new Date(post.publishedAt).toLocaleDateString()}</p>
                    {post.author && <p>Author: {post.author.name || "Unknown Author"}</p>}
                    {post.categories && post.categories.length > 0 && (
                        <div>
                            Categories:
                            <ul className="list-disc list-inside">
                                {post.categories.map((category) => (
                                    <li key={category._id}>{category.title}</li>
                                ))}
                            </ul>
                        </div>
                    )}
                </div>

                {/* Body Content */}
                <div className={styles.prose}>
                    {Array.isArray(post.body) && <PortableText value={post.body} />}
                </div>
            </main>
            <Footer />
        </>
    );
}
